// GoldbachTest.c -- a quick one-off to test Ed Pegg's results on which primes
// are 'necessary' for the Goldbach Conjecture.

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <malloc.h>

const int MaxVal = 10000000; // This is the last value we do our check up to.
const int HalfMax = MaxVal/2; // most of our arrays can/will be this size.

// Here we store the first number a prime is necessary for; any primes that
// aren't necessary will have their value here set to 0.  This has a lot of
// empty entries (all of the non-prime ones), but that's okay because we have
// plenty of memory, and it's faster than having to figure out the appropriate
// slot for a given prime.  Note that this table actually stores only the odd
// numbers/odd primes; a given value 2n+1 is stored in entry n.  (Thus, the
// first prime 3 is in slot 1; 5 is in slot 2, 7 in slot 3, 11 in slot 5,
// etc.)
int *NecessaryTable;

// Here we store the number of prime pairs that sum to a given even number.
// Just like above, this is represented in half the space; entry n maps to
// the value 2*n+2.  (Thus, to find the slot corresponding to the sum of two
// primes p=2*a+1 and q=2*b+1, we actually check index a+b.)
int *SumPairs;

// This is just your standard table of primes -- as above, note that the prime
// p=2m+1 is actually stored in this table as m.
int *Primes;

// The table we use for our sieving.
int *SieveTable;

main()
{
    int idx0, idx1, p0, p1;
    int PrimeCount;
    // Initialize our arrays
    NecessaryTable = static_cast<int *>(malloc(HalfMax*sizeof(int)));
    SumPairs = static_cast<int *>(malloc(HalfMax*sizeof(int)));
    Primes = static_cast<int *>(malloc(HalfMax*sizeof(int)));
    SieveTable = static_cast<int *>(malloc(HalfMax*sizeof(int)));
    memset(NecessaryTable, 0, HalfMax*sizeof(int));
    memset(SumPairs, 0, HalfMax*sizeof(int));

    // First, build the prime table.  Each index i in the table represents
    // 2*i+1.
    PrimeCount = 0;
    memset(SieveTable, 0, HalfMax*sizeof(int));
    for ( idx0 = 1; idx0 < HalfMax; idx0++ ) {
        if ( SieveTable[idx0] > 0 ) continue;
        p0 = 2*idx0+1;
        Primes[PrimeCount++] = idx0;
        for ( idx1 = idx0+p0; idx1 < HalfMax; idx1 += p0 )
            SieveTable[idx1] = 1;
    }

    // Now, go through and build the table of pairsums.
    for ( idx0 = 0; idx0 < PrimeCount; idx0++ ) {
        p0 = p1 = Primes[idx0];
        idx1 = idx0;
        for (idx1 = idx0;
             (idx1 < PrimeCount) && (p0+(p1=Primes[idx1]) < HalfMax);
             idx1++ )
            SumPairs[p0+p1]++;
    }

    // Next, figure out which primes are necessary.
    for ( idx0 = 0; idx0 < PrimeCount; idx0++ ) {
        // A prime is considered necessary if one of the even numbers it
        // contributes to only has one sum listed in its sumpairs, and if
        // its partner in that pair hasn't already been marked as unnecessary
        // (which would have already deducted that pair from the count).  That
        // partner can only be marked unnecessary if it's less than the
        // current prime, so we break it up into two seperate loops; one over
        // primes less than the current one, and one over primes greater.
        p0 = Primes[idx0];
        bool bNecessary = false;
        for ( idx1 = 0;
              (idx1 < idx0) && (p0+(p1=Primes[idx1]) < HalfMax);
              idx1++ ) {
            if (( SumPairs[p0+p1] == 1 ) && ( NecessaryTable[idx1] > 0 )) {
                bNecessary = true;
                NecessaryTable[idx0] = 2*(p0+p1)+2;
                break;
            }
        }
        if ( bNecessary ) continue;
        for ( idx1 = idx0;
              (idx1 < PrimeCount) && (p0+(p1=Primes[idx1]) < HalfMax);
              idx1++ ) {
            if ( SumPairs[p0+p1] == 1 ) {
                bNecessary = true;
                NecessaryTable[idx0] = 2*(p0+p1)+2;
                break;
            }
        }
        if ( bNecessary ) continue;
        // At this point, we know that the item isn't necessary; now we deduct
        // one from every pair it's part of (assuming the pair hasn't already
        // been deducted, see above).  This breaks into the same two sort of
        // loops.
        for ( idx1 = 0;
              (idx1 < idx0) && (p0+(p1=Primes[idx1]) < HalfMax);
              idx1++ ) {
            if ( NecessaryTable[idx1] > 0 ) {
                SumPairs[p0+p1]--;
            }
        }
        for ( idx1 = idx0;
              (idx1 < PrimeCount) && (p0+(p1=Primes[idx1]) < HalfMax);
              idx1++ ) {
            SumPairs[p0+p1]--;
        }
    }

    int count = 0;
    // Now, output.
    FILE *fUnNecFile = fopen("UnNecessaryFile.txt", "w");
    for ( idx0 = 0; idx0 < PrimeCount; idx0++ ) {
        if ( NecessaryTable[idx0] == 0 ) {
            fprintf(fUnNecFile, "%d", 2*Primes[idx0]+1);
            fprintf(fUnNecFile, ( ++count % 10 )?" ":"\n");
        }
    }
    fprintf(fUnNecFile, "\n");
    fclose(fUnNecFile);
    FILE *fNecFile = fopen("NecessaryFile.txt", "w");
    for ( idx0 = 0; idx0 < PrimeCount; idx0++ ) {
        if ( NecessaryTable[idx0] != 0 ) {
            fprintf(fNecFile, "%d %d\n", 2*Primes[idx0]+1, NecessaryTable[idx0]);
        }
    }
    idx1 = 0;
    for ( idx0 = 0; Primes[idx0] < HalfMax/2; idx0++ )
        if ( NecessaryTable[idx0] != 0 )
            idx1++;
    printf("Necessary ratio: %2.4f%%\n", (100.0f*static_cast<float>(idx1))/(static_cast<float>(idx0)));
}